#!/usr/bin/env bash

CUR_FILE=$BASH_SOURCE
if [ -z "$CUR_FILE" ]; then
  echo "Please use bash to execute this script."
  exit 1
fi

# GRE设备名称
GRE_DEVICE=gresv134242075
# 本机实际IP地址
GRE_REAL_LOCAL_IP=9.134.242.75
# 对端实际IP地址
GRE_REAL_PEER_IP=9.134.108.45
# GRE隧道虚拟本端地址
GRE_LOCAL_IP=99.134.242.75
# GRE隧道虚拟对端地址
GRE_PEER_IP=1.134.242.75
# GRE网卡MTU
GRE_DEVICE_MTU=1300

# 路由表ID，所有路由项都加到这个表里
ROUTE_TABLE_ID=119

# 不需要走代理的网段
CIDR_WITHOUT_PROXY=(10.84.0.0/16
10.86.0.0/15
10.80.0.0/14
10.96.0.0/14
10.88.0.0/13
10.32.0.0/12
10.64.0.0/12
10.0.0.0/11
10.125.190.16/28
10.1.136.64/28
10.1.136.16/28
10.125.201.192/26
10.125.194.0/26
10.125.202.64/26
10.125.202.192/26
10.125.195.192/26
10.125.193.192/26
9.19.178.0/26
10.85.61.64/26
10.125.197.128/26
10.12.198.64/26
10.12.199.192/26
10.12.199.0/26
10.125.199.192/26
10.12.192.0/25
10.123.96.128/25
10.125.193.0/25
10.125.194.128/25
10.125.195.0/25
10.125.197.0/25
10.123.118.0/25
10.125.199.0/25
10.125.201.0/25
10.85.61.128/25
9.30.21.0/25
172.25.32.0/24
10.125.192.0/24
172.25.36.0/24
10.123.70.0/24
10.123.97.0/24
172.25.40.0/24
10.123.140.0/24
172.25.42.0/24
10.1.164.0/24
172.25.44.0/24
172.25.46.0/24
10.6.221.0/24
10.125.200.0/24
10.85.50.0/24
9.30.20.0/24
10.20.206.0/24
10.12.153.0/24
10.125.196.0/24
10.12.154.0/24
10.6.224.0/24
10.12.126.0/24
9.77.2.0/23
172.25.34.0/23
172.25.38.0/23
10.85.48.0/23
172.25.50.0/23
10.12.62.0/23
10.123.68.0/23
10.12.90.0/23
10.12.92.0/23
10.123.98.0/23
10.123.116.0/23
10.12.124.0/23
10.125.140.0/23
10.1.162.0/23
9.19.176.0/23
9.73.196.0/23
10.20.204.0/23
10.6.206.0/23
10.6.208.0/23
10.6.222.0/23
10.125.244.0/23
10.125.248.0/23
10.12.64.0/22
10.125.136.0/22
9.154.16.0/22
10.1.152.0/22
9.77.4.0/22
10.125.240.0/22
9.73.180.0/22
9.19.184.0/22
9.19.180.0/22
10.123.64.0/22
10.12.72.0/22
9.30.16.0/22
10.123.100.0/22
9.77.104.0/22
10.123.112.0/22
10.12.120.0/22
9.73.192.0/22
9.77.8.0/21
10.12.16.0/21
9.77.96.0/21
10.123.104.0/21
10.12.112.0/21
10.125.128.0/21
10.12.136.0/21
9.73.184.0/21
10.12.232.0/21
10.24.248.0/21
9.154.0.0/20
9.30.0.0/20
10.12.96.0/20
10.12.176.0/20
9.77.16.0/20
10.85.32.0/20
10.125.224.0/20
10.85.0.0/19
9.77.32.0/19
9.77.64.0/19
10.123.0.0/18
10.125.0.0/17
9.134.0.0/15
# mnet跳板机网段
10.123.129.0/24
10.123.138.0/24
10.123.141.0/24
10.12.210.0/24
10.123.127.0/24
10.98.16.0/23
10.12.220.0/23
# devnet中使用到的保留网段
169.254.0.0/16
# oss网段
9.50.253.254/32
9.50.253.252/31
9.50.253.248/30
9.50.253.240/29
9.50.253.224/28
172.17.73.128/27
9.50.253.192/27
10.12.200.0/26
9.111.114.192/26
10.85.61.0/26
9.50.253.128/26
172.25.6.64/26
10.12.198.128/26
10.12.199.64/26
172.16.3.0/25
172.25.6.128/25
10.123.118.128/25
10.123.120.0/25
9.50.253.0/25
9.50.254.0/25
9.119.129.0/24
172.16.2.0/24
172.25.7.0/24
9.45.201.0/24
172.25.10.0/24
9.19.161.0/24
172.25.52.0/24
10.123.119.0/24
9.50.248.0/24
9.50.252.0/24
9.111.125.0/24
9.119.130.0/23
10.123.136.0/23
9.45.202.0/23
10.123.130.0/23
9.94.152.0/23
9.19.162.0/23
9.50.250.0/23
10.123.132.0/22
9.111.80.0/22
10.12.216.0/22
10.12.132.0/22
10.12.248.0/22
9.94.144.0/21
9.50.240.0/21
9.94.128.0/20
9.111.64.0/20
9.50.224.0/20
9.50.192.0/19)

# 需要走代理的网段
CIDR_WITH_PROXY=()

err_exit() {
  echo "ERROR: $@"
  exit 1
}

# 存放和该脚本相关文件的目录
PROXY_DIR=/etc/smartvpn/proxy
PROXY_SCRIPT=$PROXY_DIR/smart_proxy.sh
# 存放旧代理备份的目录
PROXY_BACK_DIR=$PROXY_DIR/backup
PROXY_BACK_FLAG="$PROXY_BACK_DIR/backup.flag"
# 备份的旧版本代理文件，键为备份出的文件名，值为备份的源文件
declare -A PROXY_BACK_FILES
PROXY_BACK_FILES=(
  [backup_profile]="/etc/profile"
  [backup_bashrc]="/etc/bashrc"
)

proxy_backup() {
  if [ -e "$PROXY_BACK_FLAG" ]; then
    return 0
  fi

  mkdir -p "$PROXY_BACK_DIR"
  for backup_file in ${!PROXY_BACK_FILES[@]}; do
    origin_file=${PROXY_BACK_FILES[$backup_file]}
    if [ ! -f "$origin_file" ]; then
      continue
    fi

    sed -i -e '/^http_proxy=/{w /dev/stdout' -e 'd}' $origin_file >> "$PROXY_BACK_DIR/$backup_file"
    sed -i -e '/^https_proxy=/{w /dev/stdout' -e 'd}' $origin_file >> "$PROXY_BACK_DIR/$backup_file"
    sed -i -e '/^ftp_proxy=/{w /dev/stdout' -e 'd}' $origin_file >> "$PROXY_BACK_DIR/$backup_file"
    sed -i -e '/^no_proxy=/{w /dev/stdout' -e 'd}' $origin_file >> "$PROXY_BACK_DIR/$backup_file"
    sed -i -e '/^#这里no_proxy/{w /dev/stdout' -e 'd}' $origin_file >> "$PROXY_BACK_DIR/$backup_file"
    sed -i -e '/^export http_proxy/{w /dev/stdout' -e 'd}' $origin_file >> "$PROXY_BACK_DIR/$backup_file"
    sed -i -e '/^#curl和wget等/{w /dev/stdout' -e 'd}' $origin_file >> "$PROXY_BACK_DIR/$backup_file"
  done
  touch "$PROXY_BACK_FLAG"

  unset http_proxy https_proxy ftp_proxy no_proxy
}

proxy_restore() {
  if [ ! -e "$PROXY_BACK_FLAG" ]; then
    return 0
  fi

  for backup_file in `ls "$PROXY_BACK_DIR"`; do
    origin_file=${PROXY_BACK_FILES[$backup_file]}
    if [ -z "$origin_file" -o ! -e "$origin_file" ]; then
      continue
    fi

    cat "$PROXY_BACK_DIR/$backup_file" >> $origin_file

    source $origin_file
  done

  rm -rf "$PROXY_BACK_DIR"
}

gre_tunnel_create() {
  if ip tunnel show "$GRE_DEVICE" &>/dev/null ; then
    ip tunnel change "$GRE_DEVICE" mode gre local "$GRE_REAL_LOCAL_IP" remote "$GRE_REAL_PEER_IP" ttl 255
  else
    ip tunnel add "$GRE_DEVICE" mode gre local "$GRE_REAL_LOCAL_IP" remote "$GRE_REAL_PEER_IP" ttl 255
  fi

  ip link set "$GRE_DEVICE" up mtu $GRE_DEVICE_MTU

  ip addr change "$GRE_LOCAL_IP" peer "$GRE_PEER_IP" dev "$GRE_DEVICE"
}

gre_tunnel_destroy() {
  ip tunnel del "$GRE_DEVICE" &>/dev/null
}

route_create() {
  # 先给不需要走代理的网段加上实际的路由
  for cidr in "${CIDR_WITHOUT_PROXY[@]}"; do
    ip route add "$cidr" via "$GRE_REAL_LOCAL_IP" table "$ROUTE_TABLE_ID"
  done
  
  # 需要走代理的网段也明确加上
  for cidr in "${CIDR_WITH_PROXY[@]}"; do
    ip route add "$cidr" dev "$GRE_DEVICE" table "$ROUTE_TABLE_ID"
  done

  # 修改默认路由为使用GRE设备
  ip route add default dev "$GRE_DEVICE" table "$ROUTE_TABLE_ID"

  if ! ip rule | grep table "$ROUTE_TABLE_ID" &>/dev/null; then 
    ip rule add table "$ROUTE_TABLE_ID"
  fi

  ip route flush cache
}

route_destroy() {
  ip rule del table "$ROUTE_TABLE_ID" 2>/dev/null
  
  ip route flush table "$ROUTE_TABLE_ID"

  ip route flush cache
}

# 现在这个脚本里所做的操作，系统重启后会丢失，所以在系统启动时也需要执行一下
boot_script_install() {
  cp -f $CUR_FILE $PROXY_SCRIPT 2>/dev/null

  if ! grep $PROXY_SCRIPT /etc/rc.d/rc.local &>/dev/null; then
    echo "bash $PROXY_SCRIPT" >> /etc/rc.d/rc.local
  fi
}

boot_script_uninstall() {
  sed -i "\|$PROXY_SCRIPT|d" /etc/rc.d/rc.local
}

install() {
  mkdir -p $PROXY_DIR

  echo "Backing up system proxy... (正在备份系统代理...)"
  proxy_backup

  echo "Configuring proxy route... (正在配置代理路由...)"
  gre_tunnel_create

  # 先销毁一下，以支持重复安装
  route_destroy
  route_create

  boot_script_install

  echo "Install finished! (安装完毕！)"
}

uninstall() {
  echo "Cleaning up proxy route... (正在清理代理路由...)"
  boot_script_uninstall

  route_destroy

  gre_tunnel_destroy

  echo "Restoring system proxy... (正在恢复系统代理...)"
  proxy_restore

  rm -rf $PROXY_DIR

  echo "Uninstall finished! (卸载完毕！)"
}

# 卸载旧版本的agent
uninstall_old_agent() {
  if [ -d "/usr/local/bin/smartvpn" ]; then
    echo "Uninstalling previous version of smartvpn agent... (正在卸载旧版本代理agent...)"
    curl -s "http://dev-smartvpn.oa.com/download/uninstall_agent.sh" | bash &>/dev/null
  fi
}

help() {
  echo """List of Commands:

install       Install smartvpn proxy (安装全流量代理)
uninstall     Uninstall smartvpn proxy (卸载全流量代理)
help          Help guide (帮助指引)
"""
}


if [ -z "$GRE_DEVICE" -o -z "$GRE_REAL_LOCAL_IP" -o -z "$GRE_REAL_PEER_IP" -o -z "$GRE_LOCAL_IP" -o -z "$GRE_PEER_IP" ]; then
  err_exit "Invalid config"
fi

operation=$1
if [ -z "$operation" ]; then
  operation="install"
fi

case "$operation" in
  install)
    uninstall_old_agent
    install
    ;;
  uninstall)
    uninstall
    ;;
  help)
    help
    ;;
  *)
    help
    exit 1
    ;;
esac
